import { colorMap } from "./colorMap";

type theme = "blue" | "green" | "red" | "orange" | "yellow";

let grey0 = "76c7d707adc03f43785e073477961fcdfec452fc";
let blue80 = "d2be57a4323e702dc5ed8d3ebd886a2e2a243b2a";
let blue100 = "1e9d29c34eb7ca20ca7a65f68580384b8087321e";
let black100 = "a508f9738ff2f93ed9d84795e4975219cd4fb107";
let yellow90 = "ebc8d2bdaeffc696ae75e5ddef629d4fcf3ac37d";
let yellow100 = "49be1121e7372bbd5596f2a6cd4f37445b54efd1";

/*
Naming convention....

r = reference, t = target

rkey ->  Reference Key
tkey ->  Target Key
rid ->   Reference Id
tid ->   Target Id
rTheme -> Reference Theme
tTheme -> Target Theme
*/

// Updates the fill based  on the new theme
function updateFill(node, tTheme:theme) {
  if (!node.fillStyleId) return;

  let rkey = getKey(node);
  let rTheme = getNamefromId(node.fillStyleId) as theme;

  let tkey = getTKey(rTheme, tTheme, rkey);

  // Handling Yellow Theme
  if (tTheme == "yellow") {

    // Handling Text
    if(node.type == "TEXT"){
      if(rkey == blue80) tkey = yellow90;
      if(rkey == blue100) tkey = yellow100;
    }

    // Handle Button Component
    if(rkey == grey0 && node.type == "TEXT" && isInsidePrimaryNode(node)) tkey = black100;
    if(isIcon(node) && isInsidePrimaryNode(node.parent)) tkey = black100;

  }

  if (!tkey) {
    console.log(`Skipping ${node.type} Fill:  ${node.name}`);
    return;
  }

  figma.importStyleByKeyAsync(tkey).then((res) => (node.fillStyleId = res.id));
}

// Updates the stroke based on the new theme
function updateStroke(node, tTheme:theme) {
  if (!node.strokeStyleId) return;

  let rkey = getKey(node);
  let rTheme = getNamefromId(node.strokeStyleId) as theme;

  let tkey = getTKey(rTheme, tTheme, rkey);

  // Handling Yellow Theme
  if(tTheme == "yellow"){
    if(isIcon(node) && isInsidePrimaryNode(node.parent)) tkey = black100;
  }

  if (!tkey) {
    console.log(`Skipping ${node.type} Stroke:  ${node.name}`);
    return;
  }

  figma
    .importStyleByKeyAsync(tkey)
    .then((res) => (node.strokeStyleId = res.id));
}



export function updateTheme(selection: SceneNode[], newTheme: theme) {
  return selection.forEach((e) => traverse(e, newTheme));

    function traverse(node, tTheme: theme) {
    // Check whether the node can be skipped for color update
    if (isSkippable(node)) return;

    console.log(`Updating: ${node.name}    ${node}`);

    // Updating Fill
    updateFill(node, tTheme);

    // Updating Stroke
    updateStroke(node, tTheme);

    // Traversing childrens
    if ("children" in node) {
      for (const child of node.children) {
        traverse(child, tTheme);
      }
    }
  }
}

// Helper Functions
// ===================================

function getNamefromId(id: string) {
  let theme = figma.getStyleById(id).name.split("/")[0].toLowerCase().trim();

  if (theme == "yellowt") theme = "yellow";
  if (theme == "oranget") theme = "orange";

  if (theme == "grey") theme = "blue";
  if (theme == "greengrey") theme = "green";
  if (theme == "redgrey") theme = "red";
  if (theme == "orangegrey") theme = "orange";
  if (theme == "yellowgrey") theme = "yellow";

  return theme;
}

function getTKey(rTheme: theme, tTheme: theme, rkey: string) {
  let cMap = colorMap.filter((e) => e[rTheme] === rkey);

  return cMap.length > 0 ? cMap[0][tTheme] : null;
}

function isSkippable(node): Boolean {
  if (node.type != "INSTANCE") return false;

  let componentsNames = [
    "_Avatar: Status",
    "Labels",
    "Ribbon",
    "Switch",
    "Status",
  ];

  return componentsNames.indexOf(node.name) !== -1 ? true : false;
}

function getKey(node){
  if(!node.fillStyleId) return null;
  return node.fillStyleId.split(":")[1].split(",")[0];
}

// return the styleId Key from the nearest parent
function getNearestParentKey(node){
  let fillStyleId = traverseup(node);
  return getKey(fillStyleId);

  function traverseup(node){
    if(node.fillStyleId) return node.fillStyleId;
    traverseup(node.parent);
  }
}

// Check wheter it is Icon node
// Note: All icon component has description as "icon" at first, then followed by other
function isIcon(node){
  if(node.parent.type == "INSTANCE" && node.parent.mainComponent.description){
   return node.parent.mainComponent.description.split(",")[0].toLowerCase() == "icon" ? true : false
  }
}

function isInsidePrimaryNode(node){
   let parentNode = figma.getNodeById(node.parent.id);
   return  ("fillStyleId" in parentNode && getKey(parentNode) == blue80) ? true : false;
}








