/**
 * Mohan Vadivel, 09/04/23
 * A simple figma plugin to lint the designs based on dotKit library.
 * ------------------------------------------------------------------------
 *
 * This plugin travers through every node in the current selection
 * and perform the following actions
 *
 * If the node is of type TEXT, then check whether text style has applied
 * to the node.
 * ├> If the node has a text style, then compare with the text style
 * │  from dotKit, if it differs then add the node to OtherLibText.
 * ├> If the node doesn't have any text style, then compare the fontName.family,
 * │  fontName.style, fontWeight and fontsize with the dotKit, if it differs then
 * │  add the node to nonStdText.
 *
 * If the node has fill or stroke property, then check whether the fillStyleId
 * and strokeStyleId has applied to the node.
 * ├> If the node has fillStyleId and there is no match for it in dotKit, then
 * │  add the node to OtherLibBg, else if the node doesn't have any fillStyleId
 * │  then check color, type and opacity in the paint style, if there is no match
 * │  for it in dotKit then add the node to nonStdBg.
 * ├> If the node has strokeStyleId and there is no match for it in dotkit, then
 * │  add the node to OtherLibStroke, else if the node doesn't have any strokeStyleId
 * │  then check the color, type and opacity in the paint style, if there is no match
 * │  for it in dotKit then add the node to nodStdStroke.
 *
 * If the node is of type Instance, then check whether the master component is from
 * dotKit, if not then add it to otherLibComponents.
 *
 * If the node has childrens, then recurs the same process to every child, till it
 * reaches the end of the tree.
 *
 * The plugin state gets reset, whenever the selection on the figma canvas get changed.
 *
 */

import libData from "./data.json";

type NodeRef = {
    name: string;
    id: string;
    type: string;
};

type LintData = {
    otherLibText: NodeRef[];
    nonStdText: NodeRef[];
    localText: NodeRef[];
    otherLibBg: NodeRef[];
    nonStdBg: NodeRef[];
    localBg: NodeRef[];
    otherLibStroke: NodeRef[];
    nonStdStroke: NodeRef[];
    localStroke: NodeRef[];
    otherLibComponent: NodeRef[];
};

let lintData: LintData = {
    nonStdText: [],
    otherLibText: [],
    localText: [],
    nonStdBg: [],
    otherLibBg: [],
    localBg: [],
    nonStdStroke: [],
    otherLibStroke: [],
    localStroke: [],
    otherLibComponent: [],
};

figma.ui.onmessage = (message) => {
    if (message.type === "LINT") {
        lintData = {
            nonStdText: [],
            otherLibText: [],
            localText: [],
            nonStdBg: [],
            otherLibBg: [],
            localBg: [],
            nonStdStroke: [],
            otherLibStroke: [],
            localStroke: [],
            otherLibComponent: [],
        };
        let currentSelection = figma.currentPage.selection;

        if (currentSelection.length === 0) {
            figma.ui.postMessage({ type: "NO_SELECTION" });
        } else {
            lintNodes(currentSelection);
            figma.ui.postMessage({ type: "DATA", data: lintData });
        }
    } else if (message.type === "FOCUS") {
        let node = figma.getNodeById(message.id) as SceneNode;
        figma.currentPage.selection = [node];
        figma.viewport.scrollAndZoomIntoView([node]);
    }
};

function clone(val: any) {
    return JSON.parse(JSON.stringify(val));
}

function lint(node: SceneNode) {
    function isSameId(id1: any, id2: any) {
        let firstId = id1.split(",")[0];
        let secondId = id2.split(",")[0];
        return firstId === secondId ? true : false;
    }

    // Instance
    if (node.type === "INSTANCE") {
        let key = node.mainComponent?.key;

        // Check for component set
        let parentComponent = node.mainComponent?.parent;
        if (parentComponent && parentComponent.type === "COMPONENT_SET") {
            key = parentComponent.key;
        }

        let componentKey = libData.components.find((e: any) => isSameId(e.key, key));
        if (!componentKey) {
            lintData.otherLibComponent.push({ name: node.name, id: node.id, type: node.type });
            return;
        } else return;
    }

    // Text Node
    typeCheck: if (node.type === "TEXT") {
        if (node.textStyleId) {
            let textStyleId = node.textStyleId;

            // Comparing with local Text styles
            let localTextStyle = figma.getLocalTextStyles();
            let localStyle = localTextStyle.find((e) => isSameId(e.id, textStyleId));
            if (localStyle) {
                lintData.localText.push({ name: node.name, id: node.id, type: node.type });
                break typeCheck;
            }

            // Comparing with dotKit Text styles
            let style = libData.textStyles.find((e: any) => isSameId(e.id, textStyleId));
            if (!style) lintData.otherLibText.push({ name: node.name, id: node.id, type: node.type });
        } else {
            let fontName = node.fontName;
            let fontSize = node.fontSize;
            let fontWeight = node.fontWeight;

            let style = libData.textStyles.find((e: any) => {
                if (
                    e.fontSize === fontSize &&
                    e.fontName.family === (fontName as FontName).family &&
                    e.fontName.style === (fontName as FontName).style
                )
                    return true;
            });
            if (!style) lintData.nonStdText.push({ name: node.name, id: node.id, type: node.type });
        }
    }

    // Fills
    fillCheck: if ("fills" in node) {
        let fills = clone(node.fills);

        if (fills.length === 0) break fillCheck;

        if (fills.length > 1) {
            lintData.nonStdBg.push({ name: node.name, id: node.id, type: node.type });
            break fillCheck;
        }

        if (node.fillStyleId) {
            let fillStyleId = node.fillStyleId;

            // Comparing with local paint styles
            let localPaintStyle = figma.getLocalPaintStyles();
            let localStyle = localPaintStyle.find((e) => isSameId(e.id, fillStyleId));
            if (localStyle) {
                lintData.localBg.push({ name: node.name, id: node.id, type: node.type });
                break fillCheck;
            }

            // Comparing with dotKit paint Styles
            let style = libData.paintStyles.find((e: any) => isSameId(e.id, fillStyleId));
            if (!style) lintData.otherLibBg.push({ name: node.name, id: node.id, type: node.type });
        } else {
            let { color, opacity, type } = fills[0];
            let style = libData.paintStyles.find((e: any) => {
                if (
                    e.paints[0].opacity === opacity &&
                    e.paints[0].type === type &&
                    e.paints[0].color.r === color.r &&
                    e.paints[0].color.g === color.g &&
                    e.paints[0].color.b === color.b
                )
                    return true;
            });
            if (!style) lintData.nonStdBg.push({ name: node.name, id: node.id, type: node.type });
        }
    }

    // Stroke
    strokeCheck: if ("strokes" in node) {
        let strokes = clone(node.strokes);

        if (strokes.length === 0) break strokeCheck;

        if (strokes.length > 1) {
            lintData.nonStdStroke.push({ name: node.name, id: node.id, type: node.type });
            break strokeCheck;
        }

        if (node.strokeStyleId) {
            let strokeStyleId = node.strokeStyleId;

            // Comparing with local paint styles
            let localPaintStyle = figma.getLocalPaintStyles();
            let localStyle = localPaintStyle.find((e) => isSameId(e.id, strokeStyleId));
            if (localStyle) {
                lintData.localStroke.push({ name: node.name, id: node.id, type: node.type });
                break strokeCheck;
            }

            // Comparing with dotKit paint styles
            let style = libData.paintStyles.find((e: any) => isSameId(e.id, strokeStyleId));
            if (!style) lintData.otherLibStroke.push({ name: node.name, id: node.id, type: node.type });
        } else {
            let { opacity, type, color } = strokes[0];
            let style = libData.paintStyles.find((e: any) => {
                if (
                    e.paints[0].opacity === opacity &&
                    e.paints[0].type === type &&
                    e.paints[0].color.r === color.r &&
                    e.paints[0].color.g === color.g &&
                    e.paints[0].color.b === color.b
                )
                    return true;
            });
            if (!style) lintData.nonStdStroke.push({ name: node.name, id: node.id, type: node.type });
        }
    }

    // Traversing into its children
    if ("children" in node && node.children.length > 0) {
        lintNodes(node.children);
    }
}

function lintNodes(nodes: readonly SceneNode[]) {
    nodes.forEach((e) => lint(e));
}

figma.showUI(__html__);
lintNodes(figma.currentPage.selection);
console.log(lintData);
